/******************************************************************************
 *  [ 1984 toc (aim) ]                                                        *
 *                                                                            *
 *  pedram amini (pedram@redhive.com)                                         *
 *  pedram.redhive.com                                                        *
 *                                                                            *
 ******************************************************************************/

#include "1984.h"

/******************************************************************************
 *  toc_parse                                                                 *
 *                                                                            *
 *  take a packet: ensure it's a valid toc packet and that we're interested   *
 *  in it, if so format and log it.                                           *
 *  arg1: (char *)            pointer to packet                               *
 *  arg2: (struct _options *) pointer to 1984 options struct                  *
 *  ret:  none                                                                *
 ******************************************************************************/
void
toc_parse (char *packet, struct _options *options)   {
    int i,
        in_tag = FALSE,                        /* within html tag boolean     */
        data_size;                             /* toc payload size            */
    struct ip     *ip;                         /* ip header                   */
    struct toc_header *toc;                    /* toc header                  */
    struct _log_info log_info;                 /* 1984 log info data struct   */
    char   *data,                              /* pointer to toc payload      */
           *start = NULL,                      /* pointer to start of message */
           *end   = NULL,                      /* pointer to end of message   */
           message[TOC_MAX_MESSAGE_LENGTH+1],  /* html stripped message       */
           screenname[TOC_MAX_SN_LENGTH+1],    /* from / to screenname        */
           command[TOC_MAX_COMMAND_LENGTH+1],  /* toc command                 */
           token;                              /* command separator           */

    memset(message,    '\0', TOC_MAX_MESSAGE_LENGTH+1);
    memset(screenname, '\0', TOC_MAX_SN_LENGTH+1);
    memset(command,    '\0', TOC_MAX_COMMAND_LENGTH+1);
    memset(&log_info,  '\0', sizeof(struct _log_info));

    ip   = (struct ip *)           (packet + options->offset);
    toc  = (struct toc_header *)   (packet + options->offset + IP_H + TCP_H);
    data = (char *)                (packet + options->offset + IP_H + TCP_H
                                           + sizeof(struct toc_header))
                                           + TOC_DATA_OFFSET;

    /* get the byte conversions out of the way */
    toc->command_start = ntohs(toc->command_start);
    ip->ip_len         = ntohs(ip->ip_len);

    /* calculate payload size */
    data_size = ip->ip_len - IP_H - TCP_H;

    /* ensure that we have data to analyze */
    if ( !data_size )
        return;

    /* mark the end of the payload (we find the start later) */
    end = (char *)toc + data_size;

    /* ensure that this is an actual toc packet */
    if (toc->command_start != TOC_COMMAND_START)
        return;

    /* extract command */
    for ( ; *data != ' ' && *data != ':'; data++)   {
        /* make sure we don't overflow this buffer */
        if (strlen(command) >= TOC_MAX_COMMAND_LENGTH)
            return;
     
        sprintf(command, "%s%c", command, *data);
    }

    /* we're only interested in the following commands: */
    if ( strcmp(command, "IM_IN")       != 0 &&
         strcmp(command, "toc_send_im") != 0 &&
         strcmp(command, "toc_signon")  != 0 )
            return;
        
    /* if we are dealing with a signon then log it and return */
    if ( strcmp(command, "toc_signon") == 0)    {
        for (data++ ; *data != ' '; data++);      /* skip server name   */
        for (data++ ; *data != ' '; data++);      /* skip server port   */
        for (data++ ; *data != ' '; data++) {    /* extract screenname */
            /* make sure we don't overflow this buffer */
            if (strlen(screenname) >= TOC_MAX_SN_LENGTH)
                return;
            
            sprintf(screenname, "%s%c", screenname, *data);
        }

        log_info.protocol = TOC;
        log_info.log_type = LI_SIGNON;

        strncpy(log_info.s_addr, inet_ntoa(ip->ip_src), LI_IP);
        strncpy(log_info.s_name, screenname, LI_NAME);

        log_signon(&log_info, options);
        return;
    }
    
    /* if we've reached this point we're dealing with a chat message */
    
    token = *data;
    data++;
    
    /* extract screenname */
    for ( ; *data != token; data++) {
        /* make sure we don't overflow this buffer */
        if (strlen(screenname) >= TOC_MAX_SN_LENGTH)
            return;
            
        sprintf(screenname, "%s%c", screenname, *data);
    }
        
    /* determine message start */
    if (token == ':')
        start = data + 3;
    else
        start = data + 1;
    
    /* extract message while ignoring html tags */
    for (data = start; data != end; data++) {
        if (*data == '<') { in_tag = TRUE;  continue; }
        if (*data == '>') { in_tag = FALSE; continue; }
        
        /* make sure we don't overflow this buffer */
        if (strlen(message) >= TOC_MAX_MESSAGE_LENGTH)
            return;
        
        /* make sure the character is printable */
        if (!in_tag && isprint(*data))
            strncat(message, data, 1);
    }

    /* fill in the log_info data struct */
    log_info.protocol = TOC;
    log_info.log_type = LI_MESSAGE;

    /* we're dealing with an incoming message */
    if ( strcmp(command, "IM_IN") == 0)    {    
        log_info.direction = LI_TO_CLIENT;
        strncpy(log_info.s_name, screenname, LI_NAME);
    }

    /* we're dealing with an outgoing message */
    if ( strcmp(command, "toc_send_im") == 0)    {
        log_info.direction = LI_FROM_CLIENT;
        strncpy(log_info.d_name, screenname, LI_NAME);
    }

    strncpy(log_info.s_addr, inet_ntoa(ip->ip_src), LI_IP);
    strncpy(log_info.d_addr, inet_ntoa(ip->ip_dst), LI_IP);

    strncpy(log_info.message, message, LI_MESSAGE);

    /* log this entry */
    log_message(&log_info, options);
}

