/******************************************************************************
 *  [ 1984 oscar (aim) ]                                                      *
 *                                                                            *
 *  pedram amini (pedram@redhive.com)                                         *
 *  pedram.redhive.com                                                        *
 *                                                                            *
 ******************************************************************************/

#include "1984.h"

/******************************************************************************
 *  oscar_parse                                                               *
 *                                                                            *
 *  take a packet: ensure it's a valid oscar packet and that we're interested *
 *  in it, if so format and log it.                                           *
 *  arg1: (char *)            pointer to packet                               *
 *  arg2: (struct _options *) pointer to 1984 options struct                  *
 *  ret:  none                                                                *
 ******************************************************************************/
void
oscar_parse (char *packet, struct _options *options)   {
    int i,
        in_tag = FALSE,                        /* within html tag boolean     */
        sn_offset;                             /* screenname offset from data */
    struct ip     *ip;                         /* ip header                   */
    struct oscar_header *osc;                  /* oscar header                */
    struct _log_info log_info;                 /* 1984 log info data struct   */
    char   *data,                              /* pointer to oscar payload    */
           *start = NULL,                      /* pointer to start of message */
           *end   = NULL,                      /* pointer to end of message   */
           message[OSCAR_MAX_MESSAGE_LENGTH+1],/* html stripped message       */
           screenname[OSCAR_MAX_SN_LENGTH+1];  /* from / to screenname        */

    memset(message,    '\0', OSCAR_MAX_MESSAGE_LENGTH+1);
    memset(screenname, '\0', OSCAR_MAX_SN_LENGTH+1);
    memset(&log_info,  '\0', sizeof(struct _log_info));

    ip   = (struct ip *)           (packet + options->offset);
    osc  = (struct oscar_header *) (packet + options->offset + IP_H + TCP_H);
    data = (char *)                (packet + options->offset + IP_H + TCP_H
                                           + sizeof(struct oscar_header));

    /* get the byte conversions out of the way */
    osc->data_length            = ntohs(osc->data_length);
    osc->fnac_family_id         = ntohs(osc->fnac_family_id);
    osc->fnac_family_subtype_id = ntohs(osc->fnac_family_subtype_id);

    /* if the data length is greater than 1018 then the message will be split
       into two packets. lets grab what we can from this packet and forget
       about the rest. */
    if (osc->data_length > 1018)
        osc->data_length = 1018;

    /* ensure that this is an actual oscar packet */
    if (osc->command_start != OSCAR_COMMAND_START)
        return;

    /* ensure that we have data to analyze */
    if ( !osc->data_length )
        return;

    /* determine the message type that we're dealing with */
    switch (osc->fnac_family_id)    {
        case OSCAR_FAMILY_MESSAGING:
            sn_offset = OSCAR_SN_OFFSET_M;
            break;
        case OSCAR_FAMILY_CHAT:
            sn_offset = OSCAR_SN_OFFSET_C;
            break;
        case OSCAR_FAMILY_SIGNON:
            sn_offset = OSCAR_SN_OFFSET_S;
            break;
        default:        /* we don't care about anything else */
            return;
    }

    /* extract the screenname */
    strncpy(screenname, (char *)(data + sn_offset), OSCAR_MAX_SN_LENGTH);

    /* cleanup dirty screenames */
    for (i = 0; i < strlen(screenname); i++)
        if ( !isprint(screenname[i]) )
            screenname[i] = '\0';

    /* if we are dealing with a signon then log it and return */
    if (osc->fnac_family_id == OSCAR_FAMILY_SIGNON)   {
        if (osc->fnac_family_subtype_id == OSCAR_SIGNON_SUBTYPE) {
            log_info.protocol = OSCAR;
            log_info.log_type = LI_SIGNON;

            strncpy(log_info.s_addr, inet_ntoa(ip->ip_src), LI_IP);
            strncpy(log_info.s_name, screenname, LI_NAME);

            log_signon(&log_info, options);
        }
        return;
    }

    /* if we've reached this point we're dealing with a chat message */

    /* determine the actual start and end of the message within data */
    (void *)start = memchr ((void *)data, '<', osc->data_length);
    
    /* determine the end of the packet. minus 4 is used to compensate for the
       last 2 fields in the oscar header data structure */
    (void *)end   = memrchr((void *)data, '>', osc->data_length) + 1;

    /* if a start was found then we have a valid message */
    if (start)
        /* strip html tags from message */
        for (data = start; data != end; data++) {
            if (*data == '<') { in_tag = TRUE;  continue; }
            if (*data == '>') { in_tag = FALSE; continue; }
            
            /* make sure we don't overflow this buffer */
            if (strlen(message) >= OSCAR_MAX_MESSAGE_LENGTH)
                return;
            
            /* make sure the character is printable */
            if (!in_tag && isprint(*data))
                strncat(message, data, 1);
        }
    /* if both a start and an end were NOT found then we return */
    else
        return;

    /* fill in the log_info data struct */
    log_info.protocol = OSCAR;
    log_info.log_type = LI_MESSAGE;

    switch (osc->fnac_family_subtype_id)    {
        case OSCAR_MSG_TO_CLIENT:
            log_info.direction = LI_TO_CLIENT;
            strncpy(log_info.s_name, screenname, LI_NAME);
            break;
        case OSCAR_MSG_FROM_CLIENT:
            log_info.direction = LI_FROM_CLIENT;
            strncpy(log_info.d_name, screenname, LI_NAME);
            break;
    }

    strncpy(log_info.s_addr, inet_ntoa(ip->ip_src), LI_IP);
    strncpy(log_info.d_addr, inet_ntoa(ip->ip_dst), LI_IP);

    strncpy(log_info.message, message, LI_MESSAGE);

    /* log this entry */
    log_message(&log_info, options);
}

