#!/usr/bin/python

#
# Bid Monkey Sniping Engine
#
# pedram amini <pedram@redhive.com> <http://pedram.redhive.com>
#

import sys
import time

# import exception handling class.
from monkeyx import *


class sniping_engine:
    ############################################################################
    ### constructor
    ###
    ### initalizes internal variables, attempts to initialize databases and
    ### randomly selects a user agent to use from the user agents database.
    ###
    ### args:    bid monkey monkey object, optional bid monkey gui object.
    ### raises:  monkeyx.LOAD on db load failure.
    ### returns: bid monkey sniping engine object.
    ###
    def __init__(self, monkey, bid_now = False, monkey_gui = None):
        # declared variables and defaults.
        self.monkey     = monkey      # monkey object.
        self.bid_now    = bid_now     # bid now flag.
        self.monkey_gui = monkey_gui  # monkey gui object.


    ############################################################################
    ### log()
    ###
    ### prints a message log message either to the console or to the log window
    ### depending on whether or not the gui was started.
    ###
    ### args: message to log, flag controlling appending of newline.
    ###
    def log(self, text, new_line = True):
        if self.monkey_gui:
            self.monkey_gui.log(text, new_line)
        else:
            if new_line:
                print text + "\n"
            else:
                print text


    ############################################################################
    ### sleep()
    ###
    ### sleeps for a given amount of time. if we have a gui we call the gui
    ### objects sleep time. otherwise we use a traditional sleep.
    ###
    ### args:   time to sleep in seconds.
    ### raises: general exception if gui sleep raises an exception.
    ###
    def sleep(self, sleep_time):
        # make sure we need to sleep.
        if sleep_time <= 0:
            return

        if self.monkey_gui:
            try:    self.monkey_gui.sleep(sleep_time)
            except: raise Exception
            return

        # xxx - consider changing to simple time.sleep()...
        # this "sleeping" loop is similar to the GUI sleeping loop. within the
        # GUI this method of "sleeping" is mandatory to prevent the interface
        # from locking up. when in console mode however we can simply sleep.

        sleep_until = int(time.time()) + sleep_time

        while True:
            time_now  = int(time.time())
            time_left = sleep_until - time_now

            # if we've waited long enough.
            if time_now > sleep_until:
                print
                return

            # break down the time left into days, hours, mins and secondss.
            days  = int( time_left / 86400)
            hours = int((time_left - days * 86400) / 3600)
            mins  = int((time_left - days * 86400 - hours * 3600) / 60)
            secs  = int( time_left - days * 86400 - hours * 3600 - mins * 60)

            # update the sleep_time widget.
            status  = "sleeping zZzZzZ: "
            status += str(days)  + " days, "
            status += str(hours) + " hours, "
            status += str(mins)  + " minutes, "
            status += str(secs)  + " seconds"

            print status + "     \r",
            sys.stdout.flush()

            # sleep for a second.
            time.sleep(1)


    ############################################################################
    ### snipe()
    ###
    ### this is the heart of the sniping engine and is the routine responsible
    ### for the timing of the actual bid placement.
    ###
    ### sniping routine synopsis:
    ###   - determine time left on auction.
    ###   - sleep for most of that time.
    ###   - re-determine time left on auction.
    ###   - "review bid".
    ###   - sleep until the last minute.
    ###   - place the actual bid.
    ###
    def snipe(self):
        # local variables.
        sleep_time        = 0
        time_before_delay = 0

        # determine how much time is left on the auction.
        try:
            self.monkey.scrape_auction_end()
        except monkeyx, x:
            self.log(x.__str__())
            return

        # current price string.
        current_price  = self.monkey.get_currency()
        current_price += " "
        current_price += "%.2f" % float(self.monkey.get_price())

        # maximum bid string.
        maximum_bid  = self.monkey.get_currency()
        maximum_bid += " "
        maximum_bid += "%.2f" % float(self.monkey.get_max_bid())

        # if we have a gui then update the widgets.
        if self.monkey_gui:
            self.monkey_gui.set_title        (self.monkey.get_title())
            self.monkey_gui.set_current_price(current_price)
            self.monkey_gui.set_user_agent   (self.monkey.get_user_agent())

        # otherwise print to the console.
        else:
            print "monkey bid summary:"
            print "    item number:   " + self.monkey.get_item_number()
            print "    auction title: " + self.monkey.get_title()
            print "    current price: " + current_price
            print "    maximum bid:   " + maximum_bid
            print "    username:      " + self.monkey.get_username()
            print "    password:      " + self.monkey.get_password()
            print "    user-agent:    " + self.monkey.get_user_agent()
            print "    safety time:   " + str(self.monkey.get_safety())
            print

        # write the auction end time to log.
        until = time.localtime(int(time.time()) + self.monkey.get_ttl_secs())
        until = time.strftime("%m/%d %H:%M.%S", until)

        msg  = "auction ends in  "
        msg += str(self.monkey.get_tl_days())  + " days "
        msg += str(self.monkey.get_tl_hours()) + " hours "
        msg += str(self.monkey.get_tl_mins())  + " minutes "
        msg += str(self.monkey.get_tl_secs())  + " seconds "
        msg += "on "
        msg += until
        self.log(msg)

        # if the bid now flag was raised then announce that fact.
        if self.bid_now:
            self.log("placing bid immediately ...")

        if not self.bid_now and self.monkey.get_tl_hours() > 0:
            self.log("sleeping until we get closer to the end of the auction.")

            # sleep until the final hour then recalculate auction end time.
            # we subtract the safety time plus a buffer to ensure that when we
            # re-awake at the final hour that we have enough time to bid.
            sleep_time  = self.monkey.get_tl_hours() * 3600
            sleep_time += self.monkey.get_tl_days()  * 86400
            sleep_time -= self.monkey.get_safety()   + 60

            # ensure that we don't sleep right to the end of the auction.
            if self.monkey.get_tl_mins() == 0:
                sleep_time -= self.monkey.get_safety() + 60

            # determine when we're going to sleep to until.
            until = time.localtime(int(time.time()) + sleep_time)
            until = time.strftime("%m/%d %H:%M.%S", until)

            # go to sleep.
            try:
                self.sleep(sleep_time)
            except:
                self.log("\nuser interrupt. someone has awoken the monkey.")
                return

            # the final hour has arrived.
            # let us redetermine when the auction ends.
            try:
                self.monkey.scrape_auction_end()
            except monkeyx, x:
                self.log(x.__str__())
                return

            msg  = "auction ends in  "
            msg += str(self.monkey.get_tl_days())  + " days "
            msg += str(self.monkey.get_tl_hours()) + " hours "
            msg += str(self.monkey.get_tl_mins())  + " minutes "
            msg += str(self.monkey.get_tl_secs())  + " seconds "
            self.log(msg)

        # store the current time to measure the following procedural delay.
        time_before_delay = int(time.time())

        # "review bid" - start the bidding process.
        try:
            self.monkey.scrape_review_bid()
        except monkeyx, x:
            self.log(x.__str__())
            return

        # if we're not supposed to bid immediately:
        if not self.bid_now:
            self.log("sniping the auction.")

            # calculate sleep time.
            sleep_time = self.monkey.get_ttl_secs() - self.monkey.get_safety()

            # determine when we're going to sleep to until.
            until = time.localtime(int(time.time()) + sleep_time)
            until = time.strftime("%m/%d %H:%M.%S", until)

            # sleep for proper amount of time.
            # take into account the procedural delay.
            # we count the procedural delay twice, once for
            # monkey.scrape_review_bid() and once more for
            # monkey.scrape_place_bid().
            sleep_time -= 2 * (int(time.time()) - time_before_delay)

            # go to sleep.
            try:
                self.sleep(sleep_time)
            except:
                self.log("\nuser interrupt. someone has awoken the monkey.")
                return

        self.log("placing bid ...")

        # place the bid.
        try:
            self.monkey.scrape_place_bid()
        except monkeyx, x:
            self.log(x.__str__())
            return

        msg  = "bid placed. "
        msg += "check your e-mail for the outcome. "
        msg += "give monkey banana.\n"
        self.log(msg)
