#!/usr/bin/python

#
# Bid Monkey GUI
#
# pedram amini <pedram@redhive.com> <http://pedram.redhive.com>
#

import sys
import time

# import sniping engine, and exception handling classes.
sys.path.append("classes")
from sniping_engine    import *
from monkeyx           import *

# attempt to import the appropriate gtk libraries.
try:
    import pygtk
    pygtk.require("2.0")
except:
    pass

try:
    import gtk
    import gtk.glade
except:
    print "import gtk/gtk.glade failed"
    sys.exit(1)


class bid_monkey_gui:
    ############################################################################
    ### constructor
    ###
    ### initializes internal variables, establishes signal handlers, and
    ### initializes the log view and the status bar.
    ###
    ### args: bid monkey object.
    ###
    def __init__(self, monkey):
        self.monkey = monkey
        self.safety = 30

        # cancel button clicked?
        self.cancel_clicked_flag = False

        # load the glade xml file.
        self.wTree  = gtk.glade.XML("gui/bid_monkey.glade", "bid_monkey")

        # establish the signal handlers.
        signal_handlers = {
            "on_help_clicked"            : self.help_clicked,
            "on_go_monkey_clicked"       : self.go_monkey_clicked,
            "on_renew_patterns_clicked"  : self.renew_patterns_clicked,
            "on_cancel_clicked"          : self.cancel_clicked,
            "on_bid_monkey_destroy"      : self.bid_monkey_destroy
        }

        self.wTree.signal_autoconnect(signal_handlers)

        # create a handle to the log buffer, log view and log iter.
        self.log_buff = gtk.TextBuffer()
        self.log_view = self.wTree.get_widget("monkey_log")
        self.log_view.set_buffer(self.log_buff)
        self.log_iter = self.log_buff.get_end_iter()

        # create a handle to the status bar.
        self.status = self.wTree.get_widget("monkey_status")
        self.status_context = self.status.get_context_id("monkey_status")

        # create a tag for bold text.
        header_tag = gtk.TextTag("bold")

        # set the bold property. 700 = pango.WEIGHT_BOLD, we don't want to
        # import all of pango for one silly constant.
        header_tag.set_property("weight", 700)

        # add the tag to the tag table.
        table = self.log_buff.get_tag_table()
        table.add(header_tag)

        # initialize the log window.
        log  = "\n[ bid monkey v%s ]"    % self.monkey.get_version()
        self.log_bold(log)

        log  = "patterns db version: %s\n" % self.monkey.get_patterns_ver()
        log += self.monkey.get_author_name() + " "
        log += self.monkey.get_author_url()  + "\n"
        self.log(log)

        # initialize the status bar.
        status = "patterns database version: " + self.monkey.get_patterns_ver()
        self.set_status(status)


    ############################################################################
    ### bid_monkey_destroy()
    ###
    ### called upon bid monkey gui destruction. shut down gtk.mainloop() and
    ### do any necessary cleanup.
    ###
    def bid_monkey_destroy(self, widget):
        gtk.main_quit()
        sys.exit(0)


    ############################################################################
    ### buttons_hide()
    ###
    ### hides all clickable interface buttons.
    ###
    def buttons_hide(self):
        self.wTree.get_widget("renew_patterns").hide()
        self.wTree.get_widget("go_monkey").hide()
        self.wTree.get_widget("help").hide()
        self.wTree.get_widget("cancel").show()
        self.update_screen()


    ############################################################################
    ### buttons_show()
    ###
    ### shows all clickable interface buttons.
    ###
    def buttons_show(self):
        self.wTree.get_widget("renew_patterns").show()
        self.wTree.get_widget("go_monkey").show()
        self.wTree.get_widget("help").show()
        self.wTree.get_widget("cancel").hide()
        self.update_screen()


    ############################################################################
    ### cancel_clicked()
    ###
    ### activates the self.cancel_clicked flag.
    ###
    def cancel_clicked(self, widget):
        self.cancel_clicked_flag = True


    ############################################################################
    ### die()
    ###
    ### handles fatal errors by printing supplied message and exiting.
    ###
    ### args: message to print.
    ###
    def die(self, message = ""):
        if message:
            print "\nMONKEY-FATAL> " + message

        sys.exit(1)


    ############################################################################
    ### go_monkey_clicked()
    ###
    ### signal handler for when "go monkey" button is clicked.
    ###
    ### args: widget.
    ###
    def go_monkey_clicked(self, widget):
        # hide action buttons.
        self.buttons_hide()

        # retrieve the widget values.
        item_number = self.wTree.get_widget("item_number").get_text()
        max_bid     = self.wTree.get_widget("maximum_bid").get_text()
        username    = self.wTree.get_widget("username").get_text()
        password    = self.wTree.get_widget("password").get_text()
        quantity    = self.wTree.get_widget("quantity").get_text()
        safety      = self.wTree.get_widget("safety_time").get_text()
        bid_now     = self.wTree.get_widget("bid_immediately").get_active()

        # parse the maximum bid. convert it to a correctly formatted string.
        try:
            max_bid = self.monkey.parse_max_bid(max_bid)
            self.wTree.get_widget("maximum_bid").set_text(max_bid)
        except monkeyx, x:
            self.log(x.__str__())
            self.buttons_show()
            return

        # store widget values within the monkey object.
        self.monkey.set_item_number( str.lstrip( str.rstrip( item_number)))
        self.monkey.set_max_bid    ( str.lstrip( str.rstrip( max_bid    )))
        self.monkey.set_username   ( str.lstrip( str.rstrip( username   )))
        self.monkey.set_password   ( str.lstrip( str.rstrip( password   )))
        self.monkey.set_quantity   ( str.lstrip( str.rstrip( quantity   )))
        self.safety =                str.lstrip( str.rstrip( safety     ))

        # ensure the provided credentials are correct.
        try:
            self.log_bold("signing in ... please be patient.")
            self.monkey.scrape_sign_in()
        except monkeyx, x:
            self.log(x.__str__())
            self.buttons_show()
            return

        # instantiate and initialize a sniping engine.
        sniper = sniping_engine(self.monkey, bid_now, self)

        # snipe the auction.
        sniper.snipe()

        # restore action buttons.
        self.buttons_show()


    ############################################################################
    ### help_clicked()
    ###
    ### signal handler for when "?" button is clicked.
    ###
    ### args: widget.
    ###
    def help_clicked(self, widget):
        self.log("\n")
        self.log_bold("item number")
        h  = "the item number that you wish to bid on. can be"
        h += " extracted from the title.\n"
        self.log(h)

        self.log_bold("maximum bid")
        h  = "the maximum amount you are willing to pay for this"
        h += " item. this is the amount that bid monkey places on its bid. if"
        h += " you win the auction the price will never exceed this amount. the"
        h += " winning price can however be less then this amount.\n"
        self.log(h)

        self.log_bold("username")
        h  = "your valid ebay username.\n"
        self.log(h)

        self.log_bold("password")
        h  = "your valid ebay password.\n"
        self.log(h)

        self.log_bold("bid immediately")
        h  = "do not snipe the auction but instead place the bid immediately.\n"
        self.log(h)

        self.log_bold("quantity")
        h  = "number of items to bid on if the auction, defaults to one. of"
        h += " course this option is dependent on whether or not the auction"
        h += " features multiple items.\n"
        self.log(h)

        self.log_bold("renew patterns database")
        h  = "the patterns database is a separate flat text file that is"
        h += " utilized in string matching and pattern parsing. in the event"
        h += " that ebay changes it's layout or urls an update of this file is"
        h += " necessary. this is a new feature in the v2 branch of bid monkey"
        h += " allowing the patterns file to be refreshed from the web so"
        h += " that only major revisions of bid monkey will require a new"
        h += " codebase release.\n"
        self.log(h)

        self.log_bold("safety time")
        h  = "time, in seconds before the auction ends, to place the bid."
        h += " defaults to 30 seconds.\n"
        self.log(h)


    ############################################################################
    ### log()
    ###
    ### prints a message to the log window.
    ###
    ### args: message to log, flag controlling appending of newline.
    ###
    def log(self, text, new_line = True):
        # append a new line if requested to do so.
        if new_line:
            text += "\n"

        # locate the end of the log buffer.
        sob, eob = self.log_buff.get_bounds()

        # insert the text.
        self.log_buff.insert(eob, text, len(text))

        # scroll to the end of the log buffer.
        sob, eob = self.log_buff.get_bounds()

        mark = self.log_buff.create_mark("end", eob, gtk.FALSE)
        self.log_view.scroll_to_mark(mark, 0.05)

        # update the screen.
        self.update_screen()


    ############################################################################
    ### log_bold()
    ###
    ### prints a bold message to the log window.
    ###
    ### args: message to log, flag controlling appending of newline.
    ###
    def log_bold(self, text, new_line = True):
        # append a new line if requested to do so.
        if new_line:
            text += "\n"

        # locate the end of the log buffer.
        sob, eob = self.log_buff.get_bounds()

        # insert the bold text.
        self.log_buff.insert_with_tags_by_name(eob, text, "bold")

        # scroll to the end of the log buffer.
        sob, eob = self.log_buff.get_bounds()

        mark = self.log_buff.create_mark("end", eob, gtk.FALSE)
        self.log_view.scroll_to_mark(mark, 0.05)

        # update the screen.
        self.update_screen()


    ############################################################################
    ### renew_patterns_clicked()
    ###
    ### signal handler for when "renew patterns" button is clicked.
    ###
    ### args: widget.
    ###
    def renew_patterns_clicked(self, widget):
        log  = "renewing patterns database file: "
        log += self.monkey.get_db_patterns() + " ... "
        self.log(log, False)

        # try to renew the patterns file.
        try:
            self.monkey.renew()
        except monkeyx, x:
            self.log(x.__str__())
            return


        self.log("done.")

        status = "patterns database version: " + self.monkey.get_patterns_ver()
        self.set_status(status)


    ############################################################################
    ### setters
    ###
    def set_status(self, text):
        self.status.push(self.status_context, text)

    def set_title(self, text):
        self.wTree.get_widget("title").set_text(text)

    def set_current_price(self, text):
        self.wTree.get_widget("current_price").set_text(text)

    def set_user_agent(self, text):
        self.wTree.get_widget("user_agent").set_text(text)


    ############################################################################
    ### sleep()
    ###
    ### sleeps for an allotted amount of time while processing screen updates.
    ### this is essential because otherwise the gui appears to lock up.
    ###
    ### args:   time to sleep in seconds.
    ### raises: general exception on cancel flag activation.
    ###
    def sleep(self, sleep_time):
        sleep_until = int(time.time()) + sleep_time

        while True:
            time_now  = int(time.time())
            time_left = sleep_until - time_now

            # if we've waited long enough.
            if time_now > sleep_until or self.cancel_clicked_flag:
                # re-initialize status bar.
                status  = "patterns database version: "
                status += self.monkey.get_patterns_ver()
                self.set_status(status)

                # if we're here because the user clicked cancel.
                if self.cancel_clicked_flag:
                    # de-activate cancel clicked flag.
                    self.cancel_clicked_flag = False

                    # raise an exception.
                    raise Exception

                return

            # break down the time left into days, hours, mins and seconds.
            days  = int( time_left / 86400)
            hours = int((time_left - days * 86400) / 3600)
            mins  = int((time_left - days * 86400 - hours * 3600) / 60)
            secs  = int( time_left - days * 86400 - hours * 3600 - mins * 60)

            # update the sleep_time widget.
            status  = "sleeping zZzZzZ: "
            status += str(days)  + " days, "
            status += str(hours) + " hours, "
            status += str(mins)  + " minutes, "
            status += str(secs)  + " seconds"

            self.set_status(status)

            # update the screen.
            self.update_screen()

            # sleep for a second.
            time.sleep(1)


    ############################################################################
    ### update_screen()
    ###
    ### loops through pending gtk events and updates the screen.
    ###
    def update_screen(self):
        for i in xrange(10000):
            while gtk.events_pending():
                gtk.mainiteration(gtk.TRUE)
