###############################################################################
# monkey_determine_auction_end
#
# synopsis:  determine how much time is left until an auction ends.
# arguments: item number
# returns:   days, hours, mins, secs

sub monkey_determine_auction_end ($$)
{
    # arguments.
    my $item_num = shift;
    my $headers  = shift;

    my (@data, $days, $flag, $hours, $i, $mins, $secs, $time_left, $url, $currency, $price, $article);

    $url = "/ws/eBayISAPI.dll?ViewItem&item=" . $item_num;
    @data = monkey_get_url("cgi.ebay.com", $url, $headers);

    # parse through the data and retrieve the time remaining on the auction.
    $flag = 0; $time_left = "";
    for ($i = 0; $i < $#data; $i++)
    {
        $_ = $data[$i];

        # auction is already over.
        monkey_error("auction has ended") if (/auction has ended/i);

        # item could not be found.
        monkey_error("item number not found") if (/Invalid Item/i);

        # extract the title of the item.
        $article = $1 if (!defined $article && /<font face=".*?" size="4"><b>(.*)<\/b><\/font>/i);

        # recording triggers. (start recording)
        $flag = 1 if (/Time left:/);
        $flag = 2 if (/Starting bid:/ || /Current bid:/ || /Price:/);

        # extract the "time left" data.
        if ($flag == 1)
        {
            $time_left = $_ if (/<font face=".*?" size="2"><b>(.*)<\/b>/i);

            # we have all the information we need so stop recording.
            $i = $#data if (/History:/i);
        }

        # extract the current price and the currency.
        if ($flag == 2)
        {
            # commas get in the way (prices > $999).
            s/,//g;

            if (/<font face=".*"\s+size="2"><b>(\w+).[\$]{0,1}(\d+\.\d\d)</)
            {
                $currency = $1;
                $price    = $2;
                $flag     = 0;
            }
        }
    } $#data = 0;

    $days = $hours = $mins = $secs = 0;

    if ($time_left =~ /(\d*) days/ ) { $days  = $1 }
    if ($time_left =~ /(\d*) hours/) { $hours = $1 }
    if ($time_left =~ /(\d*) mins/ ) { $mins  = $1 }
    if ($time_left =~ /(\d*) secs/ ) { $secs  = $1 }

    return ($days, $hours, $mins, $secs, $currency, $price, $article);
}


###############################################################################
# monkey_error
#
# synopsis:  prints an error message and quits.
# arguments: message to print.
# returns:   none, actually we bail.

sub monkey_error ($)
{
    # arguments.
    my $message = shift;

    print STDERR "\nmonkey failed: " . $message . " ... bailing.\n";
    exit;
}


###############################################################################
# monkey_get_headers
#
# synopsis:  load a set of headers pertaining to a randomly chosen user agent.
# arguments: none
# returns:   weighted random trailer for get request.

sub monkey_get_headers ()
{
    # 0 < random < 100  (with one floating point)
    $random = sprintf("%.1f", rand(101));

    $previous = 0;
    $current  = $USER_AGENTS[0][0];

    # step through the array and determine where our random number falls in
    #  the weight ranges. break on match. we need the index.
    for ($i = 0; $i < $#USER_AGENTS; $i++)
    {
        last if ($random >= $previous && $random <= $current);
        $previous = $current;
        $current += $USER_AGENTS[$i+1][0];
    }

    # return the headers we will be using.
    return ($USER_AGENTS[$i][1], $USER_AGENTS[$i][2]);
}


###############################################################################
# monkey_get_url
#
# synopsis:  formulate a get request and retrieve data from a url.
# arguments: url, headers
# returns:   array holding page contents.

sub monkey_get_url ($$$)
{
    # arguments.
    my $host    = shift;
    my $url     = shift;
    my $headers = shift;

    # locals.
    my (@data, $get_request, $i, $new_host, $new_url, $sock);

    # we use this to flag if the target page has moved.
    my $object_moved_flag = 0;

    # generate the get_request.
    $get_request  = "GET $url HTTP/1.0\r\n";
    $get_request .= $headers;
    $get_request .= "\r\n";

    # insert host into the get request.
    $get_request = eval("sprintf(\"%s\", \"$get_request\");");

    # XXX - add timeout handler.

    $sock = IO::Socket::INET->new("$host:80") || die "$host:80\n$!";
    $sock->autoflush(1);

    #print "DEBUG-http://$host>\n$get_request\n\n";
    print $sock $get_request;

    for ($i = 0; <$sock>; $i++)
    {
        $data[$i] = $_;
        $object_moved_flag++ if (/302 Object Moved/i || /302 Moved Temporarily/i);

        # the object has moved lets extract and follow it.
        if ($object_moved_flag && /Location:/i)
        {
            /http:\/\/(.*\.ebay.com).*/;
            $new_host = $1;
            $new_url  =  $_;
            $new_url  =~ s/^.*\.ebay.com//;
            @data = &monkey_get_url($new_host, $new_url, $headers);
            last;
        }
    }

    return @data;
}


###############################################################################
# monkey_initialize_db
#
# synopsis:  load user agent database into memory.
# arguments: database
# returns:   none

sub monkey_initialize_db ($)
{
    # arguments.
    my $database = shift;

    # open the database.
    open (DATABASE, $database) || die "error opening user agent database: $database\n";

    # step through the file and load entries into memory.
    for ($i = 0; <DATABASE>; $i++)
    {
        chomp;      # remove newline.
        s/#.*//;    # remove comments.
        s/^\s+//;   # remove leading whitespace.
        s/\s+$//;   # remove trailing whitespace.

        # if nothing is left then we skip to the next line.
        ($i--, next) unless length;

        ($USER_AGENTS[$i][0], $USER_AGENTS[$i][1], $USER_AGENTS[$i][2]) = split(/:/, $_, 3);
    }

    # done with the database
    close(DATABASE);
}


###############################################################################
# monkey_place_bid
#
# synopsis:  place the bid and determine how things went.
# arguments: item number, maximum bid amount, quantity, username, password,
#            ebay key, headers.
# returns:   none.

sub monkey_place_bid ($$$$$$$)
{
    # arguments.
    my $item_num = shift;
    my $max_bid  = shift;
    my $quantity = shift;
    my $username = shift;
    my $password = shift;
    my $ebay_key = shift;
    my $headers  = shift;

    my (@data, $i, $url);
    $url = "/ws/eBayISAPI.dll?MfcISAPICommand=MakeBid" .
                                    "&item="         . $item_num .
                                    "&maxbid="       . $max_bid  .
                                    "&quant="        . $quantity .
                                    "&userid="       . "default" .
                                    "&acceptbinbid=" . "0"       .
                                    "&key="          . $ebay_key .
                                    "&mode="         . "1"       .
                                    "&user="         . $username .
                                    "&pass="         . $password ;

    @data = monkey_get_url("offer.ebay.com", $url, $headers);

    # parse through the data and determine what has happened.
    for ($i = 0; $i < $#data; $i++)
    {
        $_ = $data[$i];

        # incorrect username or password.
        monkey_error("invalid username or password") if (/Password invalid/i);

        # we've been outbid.
        monkey_error("we've been outbid") if (/You have been outbid/i);

        # success!
        $i = $#data if (/Congratulations/i);
    } $#data = 0;
}


###############################################################################
# monkey_review_bid
#
# synopsis:  begins the bidding process, and retrieves the key necessary for
#            placing the final bid.
# arguments: item number, maximum amount to bid, headers.
# returns:   key.

sub monkey_review_bid ($$$)
{
    # arguments.
    my $item_num = shift;
    my $max_bid  = shift;
    my $headers  = shift;

    my (@data, $ebay_key, $i, $url);

    $url = "/ws/eBayISAPI.dll?MfcISAPICommand=MakeBid&item=" . $item_num . "&maxbid=" . $max_bid;
    @data = monkey_get_url("offer.ebay.com", $url, $headers);

    # parse through the data and retrieve they key.
    for ($i = 0; $i < $#data; $i++)
    {
        $_ = $data[$i];

        # maximum bid too low.
        if (/Your bid must be at least (.*)<\/font>/i)
        {
            monkey_error("your bid must be at least $1");
        }

        if (/name="key"/i)
        {
            $ebay_key = $_;
            $ebay_key =~ s/.*name="key" value="//;  # strip data before the key.
            $ebay_key =~ s/">.*//;                  # strip data after the key.
            $ebay_key =~ s/\$/\\\$/g;               # this gets fixed during the get_url eval.
            chomp($ebay_key);                       # strip newline.
            $i = $#data;                            # stop looping.
        }
    } $#data = 0;

    return $ebay_key;
}


###############################################################################
# monkey_ask_interactive
#
# synopsis:  interactively ask for values
# arguments: title, default value, flag
# returns:   new value

sub monkey_ask_interactive ($$$)
{
    my $title  = shift;
    my $defval = shift;
    my $flag   = shift;
    my $value  = "";

    # monkey ask interactive flags.
    my $F_VALUE_INT = 1;

    while ($value eq "")
    {
        print "$title" . (length $defval ? " (default is $defval)" : "") . ": ";
        $value = <>;
        chomp $value;

        $value = ($value eq "") ? $defval : $value;

        if ($flag == $F_VALUE_INT)
        {
            $value = int($value);
        }
    }

    return $value;
}


###############################################################################
# monkey_parse_max_bid
#
# synopsis:  make sure the max bid value makes sense.
#            - removes commas
#            - converts european style to US style
#            - examples:
#               1,250     becomes 1250
#               100,25    becomes 100.25
#               5,100,21  becomes 5100.21
# arguments: max bid
# returns:   valid max bid

sub monkey_parse_max_bid ($)
{
    my $max_bid = shift;

    $max_bid =~ s/,(\d\d)$/.$1/;
    $max_bid =~ s/,//g;

    return $max_bid;
}


###############################################################################
# monkey_usage
#
# synopsis:  print the correct usage of bid monkey to the user.
# arguments: none
# returns:   none

sub monkey_usage
{
    print STDERR "Usage: bid_monkey -i <item_num> -m <max_bid> -u <username> -p <password> [options]";
    print STDERR "\n  --ask         | -a  ask interactively for the values.";
    print STDERR "\n  --bid_now     | -n  place bid immediately (don't sleep).";
    print STDERR "\n  --daemonize   | -d  daemonize (run in background).";
    print STDERR "\n  --dialup      | -l  call get_online / get_offline scripts to connect to internet.";
    print STDERR "\n  --dialup_time | -t  <seconds> time it takes to connect to internet (default 60).";
    print STDERR "\n  --help        | -h  show this help screen.";
    print STDERR "\n  --item_num    | -i  <item #> item number you wish to bid on.";
    print STDERR "\n  --max_bid     | -b  <max bid> maximum amount your willing bid.";
    print STDERR "\n  --password    | -p  <password> your ebay password.";
    print STDERR "\n  --quantity    | -q  <quantity> quantity to bid on (default 1).";
    print STDERR "\n  --safety      | -s  <seconds> safety time, seconds before end of auction to place bid (default 120, 2 mins).";
    print STDERR "\n  --username    | -u  <username> your ebay username.";
    print STDERR "\n\n";
    exit;
}

1;
