/* dirwatch.cpp

    pedram@redhive.com

    to do:
        - add more verbose file listings
        - add look for specific perm option
        - convert whole program to c (write linked list functions in c)
        - clean the ugly

    to compile:
        g++ dirwatch.cpp -o dirwatch -DLINUX or -DSOLARIS
*/

#include <iostream>         // c++ stream operations (cout, cin)
#include <string>           // c++ string template class
#include <list>             // c++ list template class
#include <unistd.h>         // almost everything needs this
#include <pwd.h>            // getpwuid()
#include <grp.h>            // getgrgid
#include <stdio.h>          // perror(), snprintf()
#include <stdlib.h>         // getopt()
#include <dirent.h>         // readdir(), rewinddir(), closedir()
#include <sys/types.h>      // readdir(), rewinddir(), closedir(), lstat()
#include <sys/stat.h>       // lstat()
#include <strings.h>        // strcmp(), strcat()
#include <signal.h>         // sigset()

// defines
#define ADD          0
#define REM          1
#define MAXCALLS   200      // used in match()
#define OUTPUTSIZE 150      // number of chars to print

//globals (flags)
bool exitflag  = false,     // flags that it is time to exit (SIGINT caught)
     fileflag  = false,     // flags that we are watching for a specific file
     verbflag  = false,     // flags that we want detailed file information
     userflag  = false,     // flags that we are watching for a specific user
     groupflag = false,     // flags that we are watching for a specific group
     fastflag  = false,     // flags that we want to run fast
                            //  doesn't catch file removals, only additions
     suidflag  = false,     // flags that we only want to see suid files
     vfastflag = false;     // flags that we want to as fast as possible
                            //  will not check for duplicate names (ie no list)
                            //  only suggested for empty directories

// prototypes
bool check_user    (string, string);
bool check_group   (string, string);
void print_stats   (string, int);
void map_signals   (void);
void catch_signals (int);
void usage         (char **);
bool search_list   (string, list<string>);
int match          (const char *, const char *);
list<string> clean_list    (string, list<string>);


int main (int argc, char **argv)    {
    int opt;                        // used for getopt()
    string watchdir,                // directory to watch
           filename,                // specific file we are watching for
           wholename,               // filename + path for lstat()
           username,                // specific user we are watching for
           groupname;               // specific group we are watching for
    list<string> filelist;          // hold the list of files
    struct dirent *dp;              // directory entry pointer
    DIR *dirp;                      // directory pointer

    map_signals();                  // set the signals we want caught

    // parse through the command line
    while ((opt = getopt(argc, argv, "f:Fg:hu:sv")) != EOF)    {
        switch (opt)  {
            case 'f':
                filename = optarg;
                fileflag = true;
                break;
            case 'F':
                if (fastflag)
                    vfastflag = true;
                else
                    fastflag = true;
                break;
            case 'g':
                groupname = optarg;
                groupflag = true;
                break;
            case 'u':
                username = optarg;
                userflag = true;
                break;
            case 's':
                suidflag = true;
                verbflag = true;
                break;
            case 'v':
                verbflag = true;
                break;
            case 'h':
            case '?':
                usage(argv);
                exit (EXIT_FAILURE);
                break;
        }
    }

    if ((argc - optind) == 0)
        watchdir = "/tmp";
    else
        watchdir = argv[optind];

    // announce who we are and what we are doing
    cout << "\n[dirwatch]";
    cout << "\nwatching: " << watchdir;
    if (fileflag)
        cout << "\nfilename: " << filename;
    if (userflag)
        cout << "\nusername: " << username;
    if (groupflag)
        cout << "\ngroupname: " << groupname;
    if (vfastflag)
        cout << "\nvery fast mode";
    else if (fastflag)
        cout << "\nfast mode";
    // verbose takes too much time to be fast
    if (fastflag && verbflag)
        cout << "\nwarning: verbose will effect speed. disable for faster performance";
    cout << "\n\n";

    // open the specified directory
    if ((dirp = opendir(watchdir.c_str())) == NULL) {
        perror("opendir failed");
        exit (EXIT_FAILURE);
    }

    // loop until interrupted.
    while (!exitflag)   {
        while ((dp = readdir(dirp)) != NULL)    {
            if (!vfastflag) {
                if (search_list(dp->d_name, filelist))
                    continue;
            }
            else    {
                if (strcmp(dp->d_name, ".") == 0)
                    continue;
                if (strcmp(dp->d_name, "..") == 0)
                    continue;
            }
            if (fileflag)   {
                if (!match(filename.c_str(), dp->d_name))
                    continue;
            }
            if (userflag || groupflag || verbflag)  {
                wholename = watchdir;
                wholename += "/";
                wholename += dp->d_name;
            }
            if (userflag)   {
                if (!check_user(username, wholename))
                    continue;
            }
            if (groupflag)  {
                if (!check_group(groupname, wholename))
                continue;
            }
            if (!vfastflag)
                filelist.push_back(dp->d_name);
            if (verbflag)   {
                print_stats(wholename, ADD);
            }
            else
                cout << "[+] " << dp->d_name << endl;
        }
        if (!fastflag)
            filelist = clean_list(watchdir, filelist);
        rewinddir(dirp);            // go back to the start of the directory
    }

    // cleanup
    closedir(dirp);

    return (EXIT_SUCCESS);
}


// check if file belongs to a specific user
// arg1: (string) username to check for
// arg2: (string) filename to check
// rv:   (bool)   true if file belongs to user, false otherwise
bool check_user (string checkuser, string wholename)    {
    struct stat statbuf;            // holds file status set by lstat()
    struct passwd *pwent;           // user entry
    char username[16];              // holds username

    if (lstat(wholename.c_str(), &statbuf) == -1)
        return (false);

    pwent = (struct passwd *)getpwuid(statbuf.st_uid);
    strncpy(username, pwent->pw_name, sizeof(username));

    if (strcmp(username, checkuser.c_str()) == 0)
        return (true);
    else
        return (false);
}


// check if file belongs to a specific group
// arg1: (string) groupname to check for
// arg2: (string) filename to check
// rv:   (bool)   true if file belongs to group, false otherwise
bool check_group (string checkgroup, string wholename)  {
    struct stat statbuf;            // holds file status set by lstat()
    struct group  *groupent;        // group entry
    char groupname[16];             // holds username

    if (lstat(wholename.c_str(), &statbuf) == -1)
        return (false);

    groupent = (struct group  *)getgrgid(statbuf.st_gid);
    strncpy(groupname, groupent->gr_name, sizeof(groupname));

    if (strcmp(groupname, checkgroup.c_str()) == 0)
        return (true);
    else
        return (false);
}


// print file statistics (user, group, size, perms)
// arg1: (string) filename to print stats for
// arg1: (int)    status flag, add or remove
// rv:   none
void print_stats (string wholename, int status) {
    struct stat statbuf;            // holds file status set by lstat()
    struct passwd *pwent;           // user entry
    struct group  *groupent;        // group entry
    char username[16];              // holds username
    char groupname[16];             // holds groupname
    char perms[10];                 // holds file permissions
    char output[OUTPUTSIZE];        // formatted output string
    string filename;                // holds filename without path
    bool issuid = false;            // flags if current file is suid

    filename = wholename.substr(wholename.find_last_of("/") + 1);

    if (lstat(wholename.c_str(), &statbuf) == -1)   {
        cerr << "[-] " << wholename << endl;
        return;
    }

    pwent    = (struct passwd *)getpwuid(statbuf.st_uid);
    groupent = (struct group  *)getgrgid(statbuf.st_gid);

    if (pwent == NULL)
        strcpy(username, "BAD NAME");
    else
        strncpy(username,  pwent->pw_name,    sizeof(username));

    if (groupent == NULL)
        strcpy(username, "BAD GROUP");
    else
        strncpy(groupname, groupent->gr_name, sizeof(groupname));

    bzero(perms, sizeof(perms));

    if (S_ISFIFO(statbuf.st_mode))
        perms[0] = 'f';
    if (S_ISCHR(statbuf.st_mode))
        perms[0] = 'c';
    if (S_ISDIR(statbuf.st_mode))
        perms[0] = 'd';
    if (S_ISBLK(statbuf.st_mode))
        perms[0] = 'b';
    if (S_ISREG(statbuf.st_mode))
        perms[0] = '-';
    if (S_ISLNK(statbuf.st_mode))
        perms[0] = 'l';
    if (S_ISSOCK(statbuf.st_mode))
        perms[0] = 's';
    #if defined SOLARIS
        if (S_ISDOOR(statbuf.st_mode))
            perms[0] = 'p';
    #endif

    // user permissions
    ((S_IRUSR & statbuf.st_mode) == S_IRUSR) ? (perms[1] = 'r') : (perms[1] = '-');
    ((S_IWUSR & statbuf.st_mode) == S_IWUSR) ? (perms[2] = 'w') : (perms[2] = '-');
    ((S_IXUSR & statbuf.st_mode) == S_IXUSR) ? (perms[3] = 'x') : (perms[3] = '-');

    // group permissions
    ((S_IRGRP & statbuf.st_mode) == S_IRGRP) ? (perms[4] = 'r') : (perms[4] = '-');
    ((S_IWGRP & statbuf.st_mode) == S_IWGRP) ? (perms[5] = 'w') : (perms[5] = '-');
    ((S_IXGRP & statbuf.st_mode) == S_IXGRP) ? (perms[6] = 'x') : (perms[6] = '-');

    // other permissions
    ((S_IROTH & statbuf.st_mode) == S_IROTH) ? (perms[7] = 'r') : (perms[7] = '-');
    ((S_IWOTH & statbuf.st_mode) == S_IWOTH) ? (perms[8] = 'w') : (perms[8] = '-');
    ((S_IXOTH & statbuf.st_mode) == S_IXOTH) ? (perms[9] = 'x') : (perms[9] = '-');

    // suid permissions
    if ((S_ISUID & statbuf.st_mode) == S_ISUID) {
        if (perms[3] == 'x')    {
            perms[3] =  's';
            issuid = true;
           }
        else
            perms[3] =  'S';
    }

    // sgid permissions
    if ((S_ISGID & statbuf.st_mode) == S_ISGID) {
        if (perms[6] == 'x')
            perms[6] =  's';
        else
            perms[6] =  'S';
    }

    if (suidflag && !issuid)
        return;

    // format that string
    snprintf(output, sizeof(output), "%.3s %-12.10s %-8.16s %-8.16s %-8d %-.20s", (status == ADD) ? "[+]" : "[-]", perms, username, groupname, statbuf.st_size, filename.c_str());
    cout << output << endl;
}


// the semi x-platform signal mapper
void map_signals () {
    #if defined LINUX
        // map ctrl\c to catch_signals()
        if (signal(SIGINT, catch_signals) == SIG_ERR)   {
            perror ("sigset failed for SIGTERM");
            exit (EXIT_FAILURE);
        }
    #else if defined SOLARIS
        // map ctrl\c to catch_signals()
        if (sigset (SIGINT, catch_signals) == SIG_ERR)  {
            perror ("sigset failed for SIGTERM");
            exit (EXIT_FAILURE);
        }
    #endif
}


// the signal handler
void catch_signals (int signo)  {
    switch (signo)  {
        case SIGINT:
            cout << "exiting...\n";
            exitflag = true;
            break;
    }
}


// search through the list looking for a specific node
// arg1: (string) the filename we are looking for
// arg2: (list)   the list we are to look in
// rv:   (bool)   whether it exists or not
bool search_list (string entry, list<string> the_list)  {
    list<string>::const_iterator loc;

    for (loc = the_list.begin(); loc != the_list.end(); loc++)  {
        if (*loc == entry)
            return (true);
    }

    return (false);
}


// run through the list cleaning out any non-existant files
// arg1: (list)   the list to clean
// arg2: (string) the directory we are watching (for path)
// rv:   (list)   cleaned list
list<string> clean_list (string path, list<string> the_list)    {
    string wholename;               // filename + path for lstat()
    list<string>::iterator loc;     // step through the list

    for (loc = the_list.begin(); loc != the_list.end(); loc++)  {
        wholename = path;
        wholename += "/";
        wholename += *loc;

        if (access(wholename.c_str(), F_OK) == -1)  {
            if (verbflag)   {
                print_stats(wholename, REM);
            }
            else
                cout << "[-] " << *loc << endl;
            loc = the_list.erase(loc);
        }
    }
    return (the_list);
}


// check to see if two strings match using wildcards (*?)
// arg1: (const char *) name to search for using wildcards
// arg2: (const char *) name to search against
// rv:   (int)          1 on match, 0 on no match
int match(const char *mask, const char *name)   {
    int calls = 0,
        wild  = 0,
        q     = 0;
    const char *m  = mask,
               *n  = name,
               *ma = mask,
               *na = name;

    for(;;) {
        if (++calls > MAXCALLS)
            return (0);

        if (*m == '*')  {           // we've found a wildcard
            while (*m == '*')       // step through multiple wildcards
                ++m;
            wild = 1;               // we have a wildcard
            ma = m;                 // store our position along the mask
            na = n;                 // store our position along the name
        }

        if (!*m)    {               // we've reached the end of the mask
            if (!*n)                // and the end of the name
                return (1);         // hence we must have a match

            for (--m; (m > mask) && (*m == '?'); --m);

            if ((*m == '*') && (m > mask) && (m[-1] != '\\'))
                return (1);
            if (!wild)
                return (0);
            m = ma;
        }
        else if (!*n)   {
            while(*m == '*')
                ++m;
            return (*m == 0);
        }

        if ((*m == '\\') && ((m[1] == '*') || (m[1] == '?')))   {
            ++m;
            q = 1;
        }
        else {
            q = 0;
        }

        if ((tolower(*m) != tolower(*n)) && ((*m != '?') || q)) {
            if (!wild)
                return (0);
            m = ma;
            n = ++na;
        }
        else {
            if (*m)                 // if the mask hasn't ended
                ++m;                // step to the next char
            if (*n)                 // if the name hasn't ended
                ++n;                // step to the next char
        }
    }
}


void usage (char **argv)    {
    char *p, *name;
    if ((p = strrchr(argv[0], '/')) != NULL)
        name = p + 1;
    else
        name = argv[0];

    cout << "\nUsage: " << name << " [options] directory-to-watch"
    << "\n"
    << "\t-f <\"filename\"> accepts wildcards *?\n"
    << "\t-F fast mode, use twice for very fast mode\n"
    << "\t-g <groupname>\n"
    << "\t-h your looking at it\n"
    << "\t-u <username>\n"
    << "\t-s show only suid files\n"
    << "\t-v verbose file listing\n"
    << endl
    << "\t fast mode does not pick up file deletions\n"
    << "\t very fast mode is suggested only for empty directories\n"
    << endl;
}